/* -*-c++-*- 
 * This source code is proprietary of ADIT
 * Copyright (C) 2013 Advanced Driver Information Technology Joint Venture GmbH
 * All rights reserved
 *
 * Author: Guruprasad Patkar <Guruprasad.SureshPatkar@in.bosch.com>
 * Author: Rudolf Dederer <rudolf.dederer@de.bosch.com>
*/

#include <osgBatchedText/VecArray>

namespace osgBatchedText {

const float C_HALF_FLT_MAX = 65504.f;

void convertToHalfFloatVector(const std::vector<float>& src, std::vector<half_float::half>& dst)
{
   dst.clear();
   dst.reserve(src.size());
   for (std::vector<float>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(half_float::half(*itrV));
   }
}

void convertToHalfFloatVector(const std::vector<osg::Vec2f>& src, std::vector<osg::Vec2h>& dst)
{
   dst.clear();
   dst.reserve(src.size());
   for (std::vector<osg::Vec2f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(osg::Vec2h((half_float::half)itrV->x(), (half_float::half)itrV->y()));
   }
}

void convertToHalfFloatVector(const std::vector<osg::Vec2f>& src, std::vector<osg::Vec3h>& dst)
{
   static const half_float::half zeroHalf(0.0f);
   dst.clear();
   dst.reserve(src.size());
   for (std::vector<osg::Vec2f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(osg::Vec3h((half_float::half)itrV->x(), (half_float::half)itrV->y(), zeroHalf));
   }
}

void convertToHalfFloatVector(const std::vector<osg::Vec3f>& src, std::vector<osg::Vec3h>& dst)
{
   dst.clear();
   dst.reserve(src.size());
   for (std::vector<osg::Vec3f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(osg::Vec3h((half_float::half)itrV->x(), (half_float::half)itrV->y(), (half_float::half)itrV->z()));
   }
}

void convertToHalfFloatVector(const std::vector<osg::Vec4f>& src, std::vector<osg::Vec4h>& dst)
{
   dst.clear();
   dst.reserve(src.size());
   for (std::vector<osg::Vec4f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(osg::Vec4h((half_float::half)itrV->x(), (half_float::half)itrV->y(), (half_float::half)itrV->z(), (half_float::half)itrV->w()));
   }
}

float convertToScaledHalfFloatVector(const std::vector<float>& src, std::vector<half_float::half>& dst)
{
   float scaleFactor = 0.0f;

   dst.clear();
   dst.reserve(src.size());
   // determine maximum element
   for (std::vector<float>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      scaleFactor = osg::maximum(scaleFactor, fabsf(*itrV));
   }
   scaleFactor /= C_HALF_FLT_MAX;

   float invScaleFactor = ((scaleFactor == 0.0f) ? 0.0f : 1.0f / scaleFactor);

   for (std::vector<float>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(half_float::half(*itrV * invScaleFactor));
   }
   return scaleFactor;
}

float convertToScaledHalfFloatVector(const std::vector<osg::Vec2f>& src, std::vector<osg::Vec2h>& dst)
{
   float scaleFactor = 0.0f;

   dst.clear();
   dst.reserve(src.size());
   // determine maximum element
   for (std::vector<osg::Vec2f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      scaleFactor = osg::maximum(scaleFactor, osg::maximum( fabsf(itrV->x()), fabsf(itrV->y()) ));
   }
   scaleFactor /= C_HALF_FLT_MAX;

   float invScaleFactor = ((scaleFactor == 0.0f) ? 0.0f : 1.0f / scaleFactor);

   for (std::vector<osg::Vec2f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(osg::Vec2h((half_float::half)(itrV->x() * invScaleFactor), (half_float::half)(itrV->y() * invScaleFactor)));
   }
   return scaleFactor;
}

float convertToScaledHalfFloatVector(const std::vector<osg::Vec3f>& src, std::vector<osg::Vec3h>& dst)
{
   float scaleFactor = 0.0f;

   dst.clear();
   dst.reserve(src.size());
   // determine maximum element
   for (std::vector<osg::Vec3f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      scaleFactor = osg::maximum(scaleFactor, osg::maximum(osg::maximum(fabsf(itrV->x()), fabsf(itrV->y())), fabsf(itrV->z())));
   }
   scaleFactor /= C_HALF_FLT_MAX;

   float invScaleFactor = ((scaleFactor == 0.0f) ? 0.0f : 1.0f / scaleFactor);

   for (std::vector<osg::Vec3f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(osg::Vec3h((half_float::half)(itrV->x() * invScaleFactor), (half_float::half)(itrV->y() * invScaleFactor), (half_float::half)(itrV->z() * invScaleFactor)));
   }
   return scaleFactor;
}


float convertToScaledHalfFloatVector(const std::vector<osg::Vec4f>& src, std::vector<osg::Vec4h>& dst)
{
   float scaleFactor = 0.0f;

   dst.clear();
   dst.reserve(src.size());
   // determine maximum element
   for (std::vector<osg::Vec4f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      scaleFactor = osg::maximum(scaleFactor, osg::maximum(osg::maximum(osg::maximum(fabsf(itrV->x()), fabsf(itrV->y())), fabsf(itrV->z())), fabsf(itrV->w())));
   }
   scaleFactor /= C_HALF_FLT_MAX;

   float invScaleFactor = ((scaleFactor == 0.0f) ? 0.0f : 1.0f / scaleFactor);

   for (std::vector<osg::Vec4f>::const_iterator itrV = src.begin(); itrV != src.end(); ++itrV)
   {
      dst.push_back(osg::Vec4h((half_float::half)(itrV->x() * invScaleFactor), (half_float::half)(itrV->y() * invScaleFactor), (half_float::half)(itrV->z() * invScaleFactor), (half_float::half)(itrV->w() * invScaleFactor)));
   }
   return scaleFactor;
}

}
